import {
  getStereoCameraList, // 获取双目相机列表
  initStereoCamera, // 初始化双目相机
  closeStereoCamera, // 关闭双目相机
  getStereoCameraFrame, // 获取双目相机帧
  getStereoCameraRecordStatus, // 获取双目相机录制状态
  setStereoCameraExposureGain, // 设置相机曝光和增益参数
  setStereoCameraBrightness, // 设置相机亮度
  setStereoCameraContrast, // 设置相机对比度
  setStereoCameraHue, // 设置相机色调
  setStereoCameraSaturation, // 设置相机饱和度
  setStereoCameraSharpness, // 设置相机锐度
  setStereoCameraGamma, // 设置相机伽马值
  setStereoCameraWhiteBalance, // 设置相机白平衡
  setStereoCameraResolution, // 设置相机分辨率
  setStereoCameraFps, // 设置相机帧率
  setStereoCameraDepthEnabled, // 设置相机深度传感器是否启用
  setStereoCameraDepthMode, // 设置相机深度传感器模式
  startStereoCameraRecording, // 开始双目相机录制
  stopStereoCameraRecording, // 停止双目相机录制
} from '@/api/binocularCamera'
export const binocularCameraMixins = {
  data() {
    return {
      stereoCameras: [],
      stereoCamerasImage: {
        depthImage: null,//深度图像
        leftImage: null,//左目图像
        rightImage: null,//右目图像
      },
      stereoCameraisInitialized: false,
      // 双目相机设置
      stereoSettings: {
        autoExposure: true,//自动曝光
        exposure: 0,//曝光值
        gain: 0,//增益
        brightness: 4,//亮度
        contrast: 4,//对比度
        hue: 0,//色调
        saturation: 4,//饱和度
        sharpness: 4,//锐度
        gamma: 5,//伽马值
        whiteBalanceAuto: true,//自动白平衡
        whiteBalance: 40,//白平衡
        resolution: 'HD720',//分辨率
        fps: 60,//帧率
        depthEnabled: true,//深度传感器是否启用
        depthMode: 'NEURAL_PLUS'//深度传感器模式
      },
      resolutionOptions: [
        { label: 'HD720 2560×720', value: 'HD720', fpsOptions: [15, 30, 60] },
        { label: 'HD1080 3840×1080', value: 'HD1080', fpsOptions: [15, 30] },
        { label: 'HD2K 4416×1242', value: 'HD2K', fpsOptions: [15] },
        { label: 'VGA 1344×376', value: 'VGA', fpsOptions: [15, 30, 60, 100] }
      ],
      currentFpsOptions: [15, 30, 60],
      depthModes: [
        'PERFORMANCE',
        'QUALITY',
        'ULTRA',
        'NEURAL_LIGHT',
        'NEURAL',
        'NEURAL_PLUS'
      ],
      autoRefresh: false,
      refreshInterval: 5,
      interval: null,
      animationFrameId:null,
    }
  },
  methods: {
    // 获取双目相机列表
    getStereoCameraList() {
      
      const loading = this.$loading({
        lock: true,
        text: 'Loading',
        spinner: 'el-icon-loading',
        background: 'rgba(0, 0, 0, 0.7)'
      });
      if (sessionStorage.getItem('stereoCameras')) {
        this.stereoCameras = JSON.parse(sessionStorage.getItem('stereoCameras'))
        this.stereoSettings.fps = this.stereoCameras[0].camera_fps
        this.stereoSettings.resolution = this.stereoCameras[0].camera_resolution
        this.stereoSettings.depthMode = this.stereoCameras[0].depth_mode
        this.$message.success(`获取到 ${this.stereoCameras.length} 个双目相机`);
        this.activeTab = 'stereo'
        loading.close();
        let srt = sessionStorage.getItem('stereoCameraisInitialized');
        if(srt){
          this.stereoCameraisInitialized = true
        }
        return
      }
      getStereoCameraList().then(res => {
        if (res.data.length > 0) {
          this.stereoSettings.fps = res.data[0].camera_fps
          this.stereoSettings.resolution = res.data[0].camera_resolution
          this.stereoSettings.depthMode = res.data[0].depth_mode
          this.stereoCameras = res.data
          this.$message.success(`获取到 ${this.stereoCameras.length} 个双目相机`);
          this.activeTab = 'stereo'
        } else {
          this.$message.error('未检测到双目相机')
        }
      }).finally(() => {
        loading.close();
      })
    },
    //初始化相机
    initStereoCameras() {
      const loading = this.$loading({
        lock: true,
        text: 'Loading',
        spinner: 'el-icon-loading',
        background: 'rgba(0, 0, 0, 0.7)'
      });
      let param = {
        camera_ids: this.stereoCameras[0].camera_id + '',
        camear_config: {
          init_params_config: {
            camera_resolution: this.stereoSettings.resolution,
            camera_fps: this.stereoSettings.fps,
            depth_mode: this.stereoSettings.depthMode,
            coordinate_units: this.stereoCameras[0].coordinate_units,
            sdk_verbose: this.stereoCameras[0].sdk_verbose,
            sdk_verbose_log_file: this.stereoCameras[0].sdk_verbose_log_file,
            depth_minimum_distance: this.stereoCameras[0].depth_minimum_distance,
            depth_maximum_distance: this.stereoCameras[0].depth_maximum_distance,
            depth_stabilization: this.stereoCameras[0].depth_stabilization
          },
          video_params_config: {
            aec_agc: this.stereoSettings.autoExposure,
            gain: this.stereoSettings.gain,
            exposure: this.stereoSettings.exposure,
            whitebalance_auto: this.stereoSettings.whiteBalanceAuto,
            whitebalance: this.stereoSettings.whiteBalance,
            gamma: this.stereoSettings.gamma,
            sharpness: this.stereoSettings.sharpness,
            brightness: this.stereoSettings.brightness,
            contrast: this.stereoSettings.contrast,
            saturation: this.stereoSettings.saturation,
            hue: this.stereoSettings.hue,
          }
        }
      }
      initStereoCamera(param).then(res => {
        
        if(res.data.failed_cameras.length > 0){
          this.$message.error(res.data.failed_cameras[0].message)
          return
        }
        if(res.data.successful_cameras.length > 0){
          sessionStorage.setItem('stereoCameras', JSON.stringify(this.stereoCameras));
          sessionStorage.setItem('stereoCameraisInitialized',this.stereoCameraisInitialized)
          this.stereoCameraisInitialized = true
          this.$message.success(res.data.successful_cameras[0].message)
          getStereoCameraRecordStatus(this.stereoCameras[0].camera_id)
        }
      }).finally(() => {
        loading.close();
      })
    },
    //双目相机开始录制
    startStereoRecord() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        user_name: this.$store.getters.username,
        save_dir: this.dynamiSettings.savePath
      }
      startStereoCameraRecording(param).then(res => {
        this.isRecording = true;
        this.recordingStartTime = new Date();
        this.startRecordingTimer();
        this.$message.success('开始录制')
      })
    },
    //双目相机停止录制
    stopStereoRecord() {
      stopStereoCameraRecording(this.stereoCameras[0].camera_id).then(res => {
        this.isRecording = false;
        this.stopRecordingTimer();
        this.recordingStartTime = null;
        this.$message.success(`停止录制`);
        this.$message.success('停止录制')
      })
    },
    //关闭双目相机
    closeStereoCameras() {
      if(this.stereoCameras.length > 0){
        closeStereoCamera(this.stereoCameras[0].camera_id).then(res => {
          sessionStorage.removeItem('stereoCameras')
          sessionStorage.removeItem('stereoCameraisInitialized')
          if (this.interval) {
            clearInterval(this.interval)
          }
          this.stereoCameras = []
          this.stereoCameraisInitialized = false
        })
      }
      
    },
    //获取帧
    getStereoFrame() {
      // 取消任何待处理的帧请求
      if (this.animationFrameId) {
        cancelAnimationFrame(this.animationFrameId);
      }
      getStereoCameraFrame(this.stereoCameras[0].camera_id).then(res => {
        this.stereoCamerasImage.leftImage = res.data.frame_data.left_image
        this.stereoCamerasImage.rightImage = res.data.frame_data.right_image
        this.stereoCamerasImage.depthImage = res.data.frame_data.depth_image
        // 使用requestAnimationFrame批量更新所有画布
        this.animationFrameId = requestAnimationFrame(() => {
          this.$nextTick(() => {
            this.renderAllCanvases();
          });
        });
      })
    },
    // 合并的画布渲染方法
    renderAllCanvases() {
      this.renderCanvas('canvas-leftImage', this.stereoCamerasImage.leftImage);
      this.renderCanvas('canvas-rightImage', this.stereoCamerasImage.rightImage);
      this.renderCanvas('canvas-depthImage', this.stereoCamerasImage.depthImage);
    },
    // 通用的画布渲染方法
    renderCanvas(refName, imageData) {
      if (!imageData) return;
      
      const canvas = this.$refs[refName];
      if (!canvas) return;
      
      const img = new Image();
      img.onload = () => {
        const containerWidth = canvas.parentElement.clientWidth;
        const ratio = img.width / img.height;
        const height = containerWidth / ratio;

        // 创建离屏Canvas进行绘制
        const offscreenCanvas = document.createElement('canvas');
        offscreenCanvas.width = containerWidth;
        offscreenCanvas.height = height;
        const offCtx = offscreenCanvas.getContext('2d');
        offCtx.drawImage(img, 0, 0, offscreenCanvas.width, offscreenCanvas.height);

        // 将离屏Canvas绘制到目标Canvas
        canvas.width = containerWidth;
        canvas.height = height;
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(offscreenCanvas, 0, 0, canvas.width, canvas.height);
        img.src = '';
      };
      img.src = imageData;
    },
    //自动获取
    toggleAutoRefresh() {
      this.autoRefresh = !this.autoRefresh
      if (this.autoRefresh) {
        let intervalTime = 1000 / this.refreshInterval
        this.interval = setInterval(() => {
          this.getStereoFrame()
        }, intervalTime);
      } else {
        clearInterval(this.interval)
        this.interval = null;
        if (this.animationFrameId) {
          cancelAnimationFrame(this.animationFrameId);
          this.animationFrameId = null;
        }
      }
    },
    //修改曝光和增益
    handleAutoExposureChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        is_auto_exposure_gain: this.stereoSettings.autoExposure,
        exposure: this.stereoSettings.exposure,
        gain: this.stereoSettings.gain
      }
      //修改曝光和增益
      setStereoCameraExposureGain(param).then(res => {
        this.$message.success('修改曝光和增益成功')
      })
    },
    //修改亮度
    handleBrightnessChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        brightness: this.stereoSettings.brightness
      }
      setStereoCameraBrightness(param).then(res => {
        this.$message.success('修改亮度成功')
      })
    },
    //修改对比度
    handleContrastChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        contrast: this.stereoSettings.contrast
      }
      setStereoCameraContrast(param).then(res => {
        this.$message.success('修改对比度成功')
      })
    },
    //修改色调
    handleHueChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        hue: this.stereoSettings.hue
      }
      setStereoCameraHue(param).then(res => {
        this.$message.success('修改色调成功')
      })
    },
    //修改饱和度
    handleSaturationChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        saturation: this.stereoSettings.saturation
      }
      setStereoCameraSaturation(param).then(res => {
        this.$message.success('修改饱和度成功')
      })
    },
    //修改锐度
    handleSharpnessChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        sharpness: this.stereoSettings.sharpness
      }
      setStereoCameraSharpness(param).then(res => {
        this.$message.success('修改锐度成功')
      })
    },
    //修改伽马值
    handleGammaChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        gamma: this.stereoSettings.gamma
      }
      setStereoCameraGamma(param).then(res => {
        this.$message.success('修改伽马值成功')
      })
    },
    //修改白平衡
    handleWhiteBalanceChange() {
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        is_auto_whitebalance: this.stereoSettings.whiteBalanceAuto,
        whitebalance_temperature: this.stereoSettings.whiteBalance
      }
      setStereoCameraWhiteBalance(param).then(res => {
        this.$message.success('修改白平衡成功')
      })
    },
    //修改分辨率
    handleResolutionChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let list = this.resolutionOptions.filter(item => item.value == this.stereoSettings.resolution)
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        resolution: this.stereoSettings.resolution
      }
      setTimeout(() => {
        setStereoCameraResolution(param).then(res => {
          this.currentFpsOptions = list[0].fpsOptions
          this.stereoSettings.fps = this.currentFpsOptions[this.currentFpsOptions.length - 1]
          this.handleFpsChange()
          this.$message.success('修改分辨率成功')
        })
      }, 1000);
      
    },
    //修改帧率
    handleFpsChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        frame_rate: this.stereoSettings.fps
      }
      setTimeout(() => {
        setStereoCameraFps(param).then(res => {
          this.$message.success('修改帧率成功')
        })
      }, 1000);
      
    },
    //修改深度传感器是否启用
    handleDepthEnabledChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let param = { 
        camera_id: this.stereoCameras[0].camera_id,
        depth_enabled: this.stereoSettings.depthEnabled
      }
      setTimeout(() => {
        setStereoCameraDepthEnabled(param).then(res => {
          this.$message.success(`深度传感器${this.stereoSettings.depthEnabled ? '启用' : '禁用'}成功`)
        })
      }, 1000);
      
    },
    //修改深度传感器模式
    handleDepthModeChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        depth_mode: this.stereoSettings.depthMode
      }
      setTimeout(() => {
        setStereoCameraDepthMode(param).then(res => {
          this.$message.success('修改深度传感器模式成功')
        })
      }, 1000);
      
    },
  },
  beforeDestroy() {
    //关闭双目相机
    this.closeStereoCameras()
    //清除自动获取帧
    if (this.interval) {
      clearInterval(this.interval)
    }
    //清除相机缓存
    sessionStorage.removeItem('stereoCameras')
    sessionStorage.removeItem('stereoCameraisInitialized')
  }
}
