import {
  getStereoCameraList, // 获取双目相机列表
  initStereoCamera, // 初始化双目相机
  closeStereoCamera, // 关闭双目相机
  getStereoCameraFrame, // 获取双目相机帧
  getStereoCameraRecordStatus, // 获取双目相机录制状态
  setStereoCameraExposureGain, // 设置相机曝光和增益参数
  setStereoCameraBrightness, // 设置相机亮度
  setStereoCameraContrast, // 设置相机对比度
  setStereoCameraHue, // 设置相机色调
  setStereoCameraSaturation, // 设置相机饱和度
  setStereoCameraSharpness, // 设置相机锐度
  setStereoCameraGamma, // 设置相机伽马值
  setStereoCameraWhiteBalance, // 设置相机白平衡
  setStereoCameraResolution, // 设置相机分辨率
  setStereoCameraFps, // 设置相机帧率
  setStereoCameraDepthEnabled, // 设置相机深度传感器是否启用
  setStereoCameraDepthMode // 设置相机深度传感器模式
} from '@/api/binocularCamera'
export const binocularCameraMixins = {
  data() {
    return {
      stereoCameras: [],
      stereoCamerasImage: {
        depthImage: null,//深度图像
        leftImage: null,//左目图像
        rightImage: null,//右目图像
      },
      // 双目相机设置
      stereoSettings: {
        autoExposure: true,//自动曝光
        exposure: 0,//曝光值
        gain: 0,//增益
        brightness: 4,//亮度
        contrast: 4,//对比度
        hue: 0,//色调
        saturation: 4,//饱和度
        sharpness: 4,//锐度
        gamma: 5,//伽马值
        whiteBalanceAuto: true,//自动白平衡
        whiteBalance: 40,//白平衡
        resolution: 'HD720',//分辨率
        fps: 60,//帧率
        depthEnabled: true,//深度传感器是否启用
        depthMode: 'NEURAL_PLUS'//深度传感器模式
      },
      resolutionOptions: [
        { label: 'HD720 2560×720', value: 'HD720', fpsOptions: [15, 30, 60] },
        { label: 'HD1080 3840×1080', value: 'HD1080', fpsOptions: [15, 30] },
        { label: 'HD2K 4416×1242', value: 'HD2K', fpsOptions: [15] },
        { label: 'VGA 1344×376', value: 'VGA', fpsOptions: [15, 30, 60, 100] }
      ],
      currentFpsOptions: [15, 30, 60],
      depthModes: [
        'PERFORMANCE',
        'QUALITY',
        'ULTRA',
        'NEURAL_LIGHT',
        'NEURAL',
        'NEURAL_PLUS'
      ],
      autoRefresh: false,
      refreshInterval: 15,
      interval: null,
    }
  },
  methods: {
    // 获取双目相机列表
    getStereoCameraList() {
      this.activeTab = 'stereo'
      const loading = this.$loading({
        lock: true,
        text: 'Loading',
        spinner: 'el-icon-loading',
        background: 'rgba(0, 0, 0, 0.7)'
      });
      if (sessionStorage.getItem('stereoCameras')) {
        this.stereoCameras = JSON.parse(sessionStorage.getItem('stereoCameras'))
        this.stereoSettings.fps = this.stereoCameras[0].camera_fps
        this.stereoSettings.resolution = this.stereoCameras[0].camera_resolution
        this.stereoSettings.depthMode = this.stereoCameras[0].depth_mode
        this.$message.success(`获取到 ${this.stereoCameras.length} 个双目相机`);
        loading.close();
        return
      }
      getStereoCameraList().then(res => {
        console.log(res)
        if (res.data.length > 0) {
          this.stereoSettings.fps = res.data[0].camera_fps
          this.stereoSettings.resolution = res.data[0].camera_resolution
          this.stereoSettings.depthMode = res.data[0].depth_mode
          this.stereoCameras = res.data
          this.$message.success(`获取到 ${this.stereoCameras.length} 个双目相机`);
        } else {
          this.$message.error('未检测到双目相机')
        }
      }).finally(() => {
        loading.close();
      })
    },
    //初始化相机
    initStereoCameras() {
      const loading = this.$loading({
        lock: true,
        text: 'Loading',
        spinner: 'el-icon-loading',
        background: 'rgba(0, 0, 0, 0.7)'
      });
      let param = {
        camera_ids: this.stereoCameras[0].camera_id + '',
        camear_config: {
          init_params_config: {
            camera_resolution: this.stereoSettings.resolution,
            camera_fps: this.stereoSettings.fps,
            depth_mode: this.stereoSettings.depthMode,
            coordinate_units: this.stereoCameras[0].coordinate_units,
            sdk_verbose: this.stereoCameras[0].sdk_verbose,
            sdk_verbose_log_file: this.stereoCameras[0].sdk_verbose_log_file,
            depth_minimum_distance: this.stereoCameras[0].depth_minimum_distance,
            depth_maximum_distance: this.stereoCameras[0].depth_maximum_distance,
            depth_stabilization: this.stereoCameras[0].depth_stabilization
          },
          video_params_config: {
            aec_agc: this.stereoSettings.autoExposure,
            gain: this.stereoSettings.gain,
            exposure: this.stereoSettings.exposure,
            whitebalance_auto: this.stereoSettings.whiteBalanceAuto,
            whitebalance: this.stereoSettings.whiteBalance,
            gamma: this.stereoSettings.gamma,
            sharpness: this.stereoSettings.sharpness,
            brightness: this.stereoSettings.brightness,
            contrast: this.stereoSettings.contrast,
            saturation: this.stereoSettings.saturation,
            hue: this.stereoSettings.hue,
          }
        }
      }
      initStereoCamera(param).then(res => {
        console.log(res)
        sessionStorage.setItem('stereoCameras', JSON.stringify(this.stereoCameras));
        getStereoCameraRecordStatus(this.stereoCameras[0].camera_id).then(res => {
          console.log(res)
        })
      }).finally(() => {
        loading.close();
      })
    },

    //关闭双目相机
    closeStereoCameras() {
      closeStereoCamera(this.stereoCameras[0].camera_id).then(res => {
        sessionStorage.removeItem('stereoCameras')
        this.stereoCameras = []
        console.log(res)
      })
    },
    //获取帧
    getStereoFrame() {
      getStereoCameraFrame(this.stereoCameras[0].camera_id).then(res => {
        console.log(res)
        this.stereoCamerasImage.leftImage = res.data.frame_data.left_image
        this.stereoCamerasImage.rightImage = res.data.frame_data.right_image
        this.stereoCamerasImage.depthImage = res.data.frame_data.depth_image
        this.$nextTick(() => {
          this.canvasLeftImage();
          this.canvasRightImage();
          this.canvasDepthImage();
        });
      })
    },
    canvasLeftImage() {
      if (!this.stereoCamerasImage.leftImage) return;
      const canvas = this.$refs['canvas-leftImage'];
      if (!canvas) return;
      const img = new Image();
      img.onload = () => {
        // 计算保持比例的尺寸
        const containerWidth = canvas.parentElement.clientWidth;
        const ratio = img.width / img.height;
        const height = containerWidth / ratio;

        // 设置canvas尺寸
        canvas.width = containerWidth;
        canvas.height = height;

        // 绘制图像
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
        img.src = '';
      };
      img.src = this.stereoCamerasImage.leftImage;
    },
    canvasRightImage() {
      if (!this.stereoCamerasImage.rightImage) return;
      const canvas = this.$refs['canvas-rightImage'];
      if (!canvas) return;
      const img = new Image();
      img.onload = () => {
        // 计算保持比例的尺寸
        const containerWidth = canvas.parentElement.clientWidth;
        const ratio = img.width / img.height;
        const height = containerWidth / ratio;

        // 设置canvas尺寸
        canvas.width = containerWidth;
        canvas.height = height;

        // 绘制图像
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
        img.src = '';
      };
      img.src = this.stereoCamerasImage.rightImage;
    },
    canvasDepthImage() {
      if (!this.stereoCamerasImage.depthImage) return;
      const canvas = this.$refs['canvas-depthImage'];
      if (!canvas) return;
      const img = new Image();
      img.onload = () => {
        // 计算保持比例的尺寸
        const containerWidth = canvas.parentElement.clientWidth;
        const ratio = img.width / img.height;
        const height = containerWidth / ratio;

        // 设置canvas尺寸
        canvas.width = containerWidth;
        canvas.height = height;

        // 绘制图像
        const ctx = canvas.getContext('2d');
        ctx.clearRect(0, 0, canvas.width, canvas.height);
        ctx.drawImage(img, 0, 0, canvas.width, canvas.height);
        img.src = '';
      };
      img.src = this.stereoCamerasImage.depthImage;
    },
    toggleAutoRefresh() {
      console.log("123")
      this.autoRefresh = !this.autoRefresh
      if (this.autoRefresh) {
        let intervalTime = 1000 / this.refreshInterval
        this.interval = setInterval(() => {
          this.getStereoFrame()
        }, intervalTime);
      } else {
        clearInterval(this.interval)
        this.interval = null;
      }
    },
    //修改曝光和增益
    handleAutoExposureChange() {
      console.log("修改曝光和增益")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        is_auto_exposure_gain: this.stereoSettings.autoExposure,
        exposure: this.stereoSettings.exposure,
        gain: this.stereoSettings.gain
      }
      // console.log(param)
      //修改曝光和增益
      setStereoCameraExposureGain(param).then(res => {
        console.log(res)
      })
    },
    //修改亮度
    handleBrightnessChange() {
      console.log("修改亮度")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        brightness: this.stereoSettings.brightness
      }
      setStereoCameraBrightness(param).then(res => {
        console.log(res)
      })
    },
    //修改对比度
    handleContrastChange() {
      console.log("修改对比度")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        contrast: this.stereoSettings.contrast
      }
      setStereoCameraContrast(param).then(res => {
        console.log(res)
      })
    },
    //修改色调
    handleHueChange() {
      console.log("修改色调")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        hue: this.stereoSettings.hue
      }
      setStereoCameraHue(param).then(res => {
        console.log(res)
      })
    },
    //修改饱和度
    handleSaturationChange() {
      console.log("修改饱和度")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        saturation: this.stereoSettings.saturation
      }
      setStereoCameraSaturation(param).then(res => {
        console.log(res)
      })
    },
    //修改锐度
    handleSharpnessChange() {
      console.log("修改锐度")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        sharpness: this.stereoSettings.sharpness
      }
      setStereoCameraSharpness(param).then(res => {
        console.log(res)
      })
    },
    //修改伽马值
    handleGammaChange() {
      console.log("修改伽马值")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        gamma: this.stereoSettings.gamma
      }
      setStereoCameraGamma(param).then(res => {
        console.log(res)
      })
    },
    //修改白平衡
    handleWhiteBalanceChange() {
      console.log("修改白平衡")
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        is_auto_whitebalance: this.stereoSettings.whiteBalanceAuto,
        whitebalance_temperature: this.stereoSettings.whiteBalance
      }
      setStereoCameraWhiteBalance(param).then(res => {
        console.log(res)
      })
    },
    //修改分辨率
    handleResolutionChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      this.list = this.resolutionOptions.filter(item => item.value == this.stereoSettings.resolution)
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        resolution: this.stereoSettings.resolution
      }
      setStereoCameraResolution(param).then(res => {
        console.log(res)
        this.currentFpsOptions = this.list.fpsOptions
        this.stereoSettings.fps = this.currentFpsOptions[this.currentFpsOptions.length - 1]
        this.handleFpsChange()
      })
    },
    //修改帧率
    handleFpsChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        frame_rate: this.stereoSettings.fps
      }
      setStereoCameraFps(param).then(res => {
        console.log(res)
      })
    },
    //修改深度传感器是否启用
    handleDepthEnabledChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        depth_enabled: this.stereoSettings.depthEnabled
      }
      setStereoCameraDepthEnabled(param).then(res => {
        console.log(res)
      })
    },
    //修改深度传感器模式
    handleDepthModeChange() {
      if (this.interval) {
        clearInterval(this.interval)
        this.autoRefresh = !this.autoRefresh
      }
      let param = {
        camera_id: this.stereoCameras[0].camera_id,
        depth_mode: this.stereoSettings.depthMode
      }
      setStereoCameraDepthMode(param).then(res => {
        console.log(res)
      })
    },



  },
  beforeDestroy() {
    //关闭双目相机
    this.closeStereoCameras()
    //清除自动获取帧
    if (this.interval) {
      clearInterval(this.interval)
    }
    //清除相机缓存
    sessionStorage.removeItem('stereoCameras')
  }
}
