'use strict';

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.run = run;
exports.option = option;
exports.options = options;
exports.help = help;

var _child_process = require('child_process');

var _path = require('path');

var _path2 = _interopRequireDefault(_path);

var _common = require('./common');

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

const loggerAlias = _common.logger;


function runSync(command, options) {
  try {
    const nextOptions = {
      cwd: options.cwd,
      env: options.env,
      stdio: options.stdio,
      timeout: options.timeout
    };
    const buffer = (0, _child_process.execSync)(command, nextOptions);
    if (buffer) {
      return buffer.toString();
    }
    return null;
  } catch (error) {
    throw new _common.RunJSError(error.message);
  }
}

function runAsync(command, options) {
  return new Promise((resolve, reject) => {
    const nextOptions = {
      cwd: options.cwd,
      env: options.env,
      stdio: options.stdio,
      shell: true
    };
    const asyncProcess = (0, _child_process.spawn)(command, nextOptions);
    let output = null;

    asyncProcess.on('error', error => {
      reject(new Error(`Failed to start command: ${command}; ${error.toString()}`));
    });

    asyncProcess.on('close', exitCode => {
      if (exitCode === 0) {
        resolve(output);
      } else {
        reject(new Error(`Command failed: ${command} with exit code ${exitCode}`));
      }
    });

    if (options.stdio === 'pipe') {
      asyncProcess.stdout.on('data', buffer => {
        output = buffer.toString();
      });
    }

    if (options.timeout) {
      setTimeout(() => {
        asyncProcess.kill();
        reject(new Error(`Command timeout: ${command}`));
      }, options.timeout);
    }
  });
}

function run(command, options = {}, logger = loggerAlias) {
  const binPath = _path2.default.resolve('./node_modules/.bin');

  // Pick relevant option keys and set default values
  const nextOptions = {
    env: options.env || process.env,
    cwd: options.cwd,
    async: !!options.async,
    stdio: options.stdio || 'inherit',
    timeout: options.timeout
  };

  const env = nextOptions.env;

  // Include in PATH node_modules bin path
  if (env) {
    env.PATH = [binPath, env.PATH || process.env.PATH].join(_path2.default.delimiter);
  }

  logger.title(command);

  // Handle async call
  if (options.async) {
    return runAsync(command, nextOptions);
  }

  // Handle sync call by default
  return runSync(command, nextOptions);
}

/**
 * @deprecated
 */
function option(thisObj, name) {
  return thisObj && thisObj.options && thisObj.options[name] || null;
}

function options(thisObj) {
  return thisObj && thisObj.options || {};
}

function help(func, annotation) {
  func.help = annotation;
}